/******************************************************************************

  Copyright (c) 2013-2019, Intel Corporation
  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions are met:

   1. Redistributions of source code must retain the above copyright notice,
      this list of conditions and the following disclaimer.

   2. Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.

   3. Neither the name of the Intel Corporation nor the names of its
      contributors may be used to endorse or promote products derived from
      this software without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
  ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
  LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
  SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
  CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
  ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
  POSSIBILITY OF SUCH DAMAGE.

******************************************************************************/
/*$FreeBSD$*/
/**
 * @file iavf_osdep.c
 * @brief OS compatibility layer
 *
 * Contains definitions for various functions used to provide an OS
 * independent layer for sharing code between drivers on different operating
 * systems.
 */
#include <machine/stdarg.h>

#include "iavf.h"

/********************************************************************
 * Manage DMA'able memory.
 *******************************************************************/
static void
iavf_dmamap_cb(void *arg, bus_dma_segment_t * segs, int nseg, int error)
{
        if (error)
                return;
        *(bus_addr_t *) arg = segs->ds_addr;
        return;
}

enum iavf_status
iavf_allocate_virt_mem(struct iavf_hw *hw, struct iavf_virt_mem *mem, u32 size)
{
	mem->va = malloc(size, M_IAVF, M_NOWAIT | M_ZERO);
	return(mem->va == NULL);
}

enum iavf_status
iavf_free_virt_mem(struct iavf_hw *hw, struct iavf_virt_mem *mem)
{
	free(mem->va, M_IAVF);
	mem->va = NULL;

	return(0);
}

enum iavf_status
iavf_allocate_dma_mem(struct iavf_hw *hw, struct iavf_dma_mem *mem,
	enum iavf_memory_type type __unused, u64 size, u32 alignment)
{
	device_t	dev = ((struct iavf_osdep *)hw->back)->dev;
	int		err;


	err = bus_dma_tag_create(bus_get_dma_tag(dev),	/* parent */
			       alignment, 0,	/* alignment, bounds */
			       BUS_SPACE_MAXADDR,	/* lowaddr */
			       BUS_SPACE_MAXADDR,	/* highaddr */
			       NULL, NULL,	/* filter, filterarg */
			       size,	/* maxsize */
			       1,	/* nsegments */
			       size,	/* maxsegsize */
			       BUS_DMA_ALLOCNOW, /* flags */
			       NULL,	/* lockfunc */
			       NULL,	/* lockfuncarg */
			       &mem->tag);
	if (err != 0) {
		device_printf(dev,
		    "iavf_allocate_dma: bus_dma_tag_create failed, "
		    "error %u\n", err);
		goto fail_0;
	}
	err = bus_dmamem_alloc(mem->tag, (void **)&mem->va,
			     BUS_DMA_NOWAIT | BUS_DMA_ZERO, &mem->map);
	if (err != 0) {
		device_printf(dev,
		    "iavf_allocate_dma: bus_dmamem_alloc failed, "
		    "error %u\n", err);
		goto fail_1;
	}
	err = bus_dmamap_load(mem->tag, mem->map, mem->va,
			    size,
			    iavf_dmamap_cb,
			    &mem->pa,
			    BUS_DMA_NOWAIT);
	if (err != 0) {
		device_printf(dev,
		    "iavf_allocate_dma: bus_dmamap_load failed, "
		    "error %u\n", err);
		goto fail_2;
	}
	mem->nseg = 1;
	mem->size = size;
	bus_dmamap_sync(mem->tag, mem->map,
	    BUS_DMASYNC_PREREAD|BUS_DMASYNC_PREWRITE);
	return (0);
fail_2:
	bus_dmamem_free(mem->tag, mem->va, mem->map);
fail_1:
	bus_dma_tag_destroy(mem->tag);
fail_0:
	mem->map = NULL;
	mem->tag = NULL;
	return (err);
}

enum iavf_status
iavf_free_dma_mem(struct iavf_hw *hw, struct iavf_dma_mem *mem)
{
	bus_dmamap_sync(mem->tag, mem->map,
	    BUS_DMASYNC_POSTREAD | BUS_DMASYNC_POSTWRITE);
	bus_dmamap_unload(mem->tag, mem->map);
	bus_dmamem_free(mem->tag, mem->va, mem->map);
	bus_dma_tag_destroy(mem->tag);
	return (0);
}

void
iavf_init_spinlock(struct iavf_spinlock *lock)
{
	mtx_init(&lock->mutex, "mutex",
	    "iavf spinlock", MTX_DEF | MTX_DUPOK);
}

void
iavf_acquire_spinlock(struct iavf_spinlock *lock)
{
	mtx_lock(&lock->mutex);
}

void
iavf_release_spinlock(struct iavf_spinlock *lock)
{
	mtx_unlock(&lock->mutex);
}

void
iavf_destroy_spinlock(struct iavf_spinlock *lock)
{
	if (mtx_initialized(&lock->mutex))
		mtx_destroy(&lock->mutex);
}

/*
 * Helper function for debug statement printing
 */
void
iavf_debug_shared(struct iavf_hw *hw, uint64_t mask, char *fmt, ...)
{
	va_list args;
	device_t dev;

	if (!(mask & ((struct iavf_hw *)hw)->debug_mask))
		return;

	dev = ((struct iavf_osdep *)hw->back)->dev;

	/* Re-implement device_printf() */
	device_print_prettyname(dev);
	va_start(args, fmt);
	vprintf(fmt, args);
	va_end(args);
}

u16
iavf_read_pci_cfg(struct iavf_hw *hw, u32 reg)
{
        u16 value;

        value = pci_read_config(((struct iavf_osdep *)hw->back)->dev,
            reg, 2);

        return (value);
}

void
iavf_write_pci_cfg(struct iavf_hw *hw, u32 reg, u16 value)
{
        pci_write_config(((struct iavf_osdep *)hw->back)->dev,
            reg, value, 2);

        return;
}

/**
 * rd32 - Read a 32bit hardware register value
 * @hw: the private hardware structure
 * @reg: register address to read
 *
 * Read the specified 32bit register value from BAR0 and return its contents.
 */
inline uint32_t
rd32(struct iavf_hw *hw, uint32_t reg)
{
	struct iavf_osdep *osdep = (struct iavf_osdep *)hw->back;

	KASSERT(reg < osdep->mem_bus_space_size,
	    ("iavf: register offset %#jx too large (max is %#jx)",
	    (uintmax_t)reg, (uintmax_t)osdep->mem_bus_space_size));

	return (bus_space_read_4(osdep->mem_bus_space_tag,
	    osdep->mem_bus_space_handle, reg));
}

/**
 * wr32 - Write a 32bit hardware register
 * @hw: the private hardware structure
 * @reg: the register address to write to
 * @val: the 32bit value to write
 *
 * Write the specified 32bit value to a register address in BAR0.
 */
inline void
wr32(struct iavf_hw *hw, uint32_t reg, uint32_t val)
{
	struct iavf_osdep *osdep = (struct iavf_osdep *)hw->back;

	KASSERT(reg < osdep->mem_bus_space_size,
	    ("iavf: register offset %#jx too large (max is %#jx)",
	    (uintmax_t)reg, (uintmax_t)osdep->mem_bus_space_size));

	bus_space_write_4(osdep->mem_bus_space_tag,
	    osdep->mem_bus_space_handle, reg, val);
}

inline void
iavf_flush(struct iavf_hw *hw)
{
	struct iavf_osdep *osdep = (struct iavf_osdep *)hw->back;

	rd32(hw, osdep->flush_reg);
}

void
iavf_debug_core(device_t dev, u32 enabled_mask, u32 mask, char *fmt, ...)
{
	va_list args;

	if (!(mask & enabled_mask))
		return;

	/* Re-implement device_printf() */
	device_print_prettyname(dev);
	va_start(args, fmt);
	vprintf(fmt, args);
	va_end(args);
}
